<?php
require_once '../CBDD/config.php';

// Verificar si el usuario está logueado
if (!isset($_SESSION['usuario_id'])) {
    header('Location: ../index.php');
    exit;
}

$mensaje = '';
$tipo_mensaje = '';

// Procesar subida de foto
if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_FILES['foto_perfil'])) {

    // Solo Premium puede subir foto
    if ($_SESSION['usuario_plan'] !== 'premium') {
        $mensaje = 'Solo usuarios Premium pueden subir foto de perfil';
        $tipo_mensaje = 'error';
    } else {
        $file = $_FILES['foto_perfil'];

        if ($file['error'] === 0) {

            $allowedExt = ['jpg', 'jpeg', 'png', 'gif', 'webp'];
            $maxFileSize = 5 * 1024 * 1024; // 5MB
            $filename = $file['name'];
            $filesize = $file['size'];
            $file_ext = strtolower(pathinfo($filename, PATHINFO_EXTENSION));

            // Validar tamaño
            if ($filesize > $maxFileSize) {
                $mensaje = 'El archivo es demasiado grande (máximo 5MB)';
                $tipo_mensaje = 'error';
            }
            // Validar formato
            elseif (!in_array($file_ext, $allowedExt)) {
                $mensaje = 'Formato de archivo no permitido (solo JPG, PNG, GIF, WEBP)';
                $tipo_mensaje = 'error';
            } 
            // Validar que sea una imagen real
            elseif (!getimagesize($file['tmp_name'])) {
                $mensaje = 'El archivo no es una imagen válida';
                $tipo_mensaje = 'error';
            }
            else {

                // Crear carpeta si no existe
                if (!file_exists(UPLOAD_DIR)) {
                    mkdir(UPLOAD_DIR, 0755, true);
                }

                // Eliminar foto anterior si existe
                if (isset($_SESSION['usuario_foto']) && $_SESSION['usuario_foto']) {
                    $old_photo = UPLOAD_DIR . $_SESSION['usuario_foto'];
                    if (file_exists($old_photo)) {
                        unlink($old_photo);
                    }
                }

                // Nombre único con sanitización
                $new_filename = 'user_' . $_SESSION['usuario_id'] . '_' . uniqid() . '.' . $file_ext;
                $destination = UPLOAD_DIR . $new_filename;

                // Mover archivo
                if (move_uploaded_file($file['tmp_name'], $destination)) {

                    // Optimizar imagen (redimensionar si es muy grande)
                    try {
                        $image_info = getimagesize($destination);
                        if ($image_info[0] > 800 || $image_info[1] > 800) {
                            // Aquí podrías agregar código para redimensionar la imagen
                        }
                    } catch (Exception $e) {
                        // Continuar si falla la optimización
                    }

                    try {
                        $conn = getConnection();
                        $stmt = $conn->prepare("UPDATE usuarios SET foto_perfil = ?, fecha_actualizacion = NOW() WHERE id = ?");
                        $stmt->execute([$new_filename, $_SESSION['usuario_id']]);

                        $_SESSION['usuario_foto'] = $new_filename;

                        $mensaje = 'Foto de perfil actualizada exitosamente';
                        $tipo_mensaje = 'exito';

                    } catch (PDOException $e) {
                        $mensaje = 'Error al actualizar foto: ' . $e->getMessage();
                        $tipo_mensaje = 'error';
                        // Eliminar archivo si falla la BD
                        if (file_exists($destination)) {
                            unlink($destination);
                        }
                    }

                } else {
                    $mensaje = 'Error al subir el archivo';
                    $tipo_mensaje = 'error';
                }
            }

        } else {
            $errores = [
                UPLOAD_ERR_INI_SIZE => 'El archivo excede el tamaño máximo permitido',
                UPLOAD_ERR_FORM_SIZE => 'El archivo excede el tamaño del formulario',
                UPLOAD_ERR_PARTIAL => 'El archivo se subió parcialmente',
                UPLOAD_ERR_NO_FILE => 'No se seleccionó ningún archivo',
                UPLOAD_ERR_NO_TMP_DIR => 'Falta carpeta temporal',
                UPLOAD_ERR_CANT_WRITE => 'Error al escribir el archivo',
                UPLOAD_ERR_EXTENSION => 'Extensión de PHP detuvo la subida'
            ];
            $mensaje = $errores[$file['error']] ?? 'Error desconocido al subir el archivo';
            $tipo_mensaje = 'error';
        }
    }
}

// Obtener estadísticas del usuario
$stats = [
    'conversaciones' => 0,
    'dias_uso' => 0,
    'ultima_sesion' => 'Nunca'
];

try {
    $conn = getConnection();
    
    // Contar conversaciones (si tienes tabla de conversaciones)
    $stmt = $conn->prepare("SELECT COUNT(*) as total FROM conversaciones WHERE usuario_id = ?");
    $stmt->execute([$_SESSION['usuario_id']]);
    $stats['conversaciones'] = $stmt->fetch(PDO::FETCH_ASSOC)['total'] ?? 0;
    
    // Calcular días de uso
    $stmt = $conn->prepare("SELECT DATEDIFF(NOW(), fecha_registro) as dias FROM usuarios WHERE id = ?");
    $stmt->execute([$_SESSION['usuario_id']]);
    $stats['dias_uso'] = $stmt->fetch(PDO::FETCH_ASSOC)['dias'] ?? 0;
    
} catch (PDOException $e) {
    // Si hay error, mantener valores por defecto
}
?>

<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Dashboard - Aurora | Tu Asistente de Apoyo Emocional</title>
    <meta name="description" content="Panel de control de Aurora - Tu asistente virtual de apoyo emocional">
    <link rel="preconnect" href="https://fonts.googleapis.com">
    <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="../CSS/dash.css">
</head>
<body>
    <div class="container">
        <!-- Header -->
        <div class="header">
            <div class="user-info">
                <div class="foto-container">
                    <?php if (isset($_SESSION['usuario_foto']) && $_SESSION['usuario_foto']): ?>
                        <img src="<?php echo UPLOAD_DIR . htmlspecialchars($_SESSION['usuario_foto']); ?>" 
                             alt="Foto de perfil" 
                             class="foto-perfil"
                             loading="lazy">
                    <?php else: ?>
                        <div class="default-avatar">👤</div>
                    <?php endif; ?>
                </div>
                
                <div class="user-details">
                    <h2><?php echo htmlspecialchars($_SESSION['usuario_nombre']); ?></h2>
                    <div class="user-meta">
                        <span class="badge <?php echo $_SESSION['usuario_plan']; ?>">
                            <?php echo strtoupper($_SESSION['usuario_plan']); ?>
                        </span>
                        <div class="user-stats">
                            <span class="stat-item">
                                <strong><?php echo $stats['dias_uso']; ?></strong> días contigo
                            </span>
                        </div>
                    </div>
                </div>
            </div>
            
            <div class="header-actions">
                <a href="logout.php" class="logout-btn">Cerrar Sesión</a>
            </div>
        </div>
        
        <!-- Mensajes -->
        <?php if ($mensaje): ?>
            <div class="mensaje <?php echo $tipo_mensaje; ?>">
                <?php echo htmlspecialchars($mensaje); ?>
            </div>
        <?php endif; ?>
        
        <!-- Contenido Principal -->
        <div class="content">
            <!-- Card Principal -->
            <div class="card">
                <div class="card-header">
                    <span class="card-icon">🌟</span>
                    <h3>Bienvenido a Aurora</h3>
                </div>
                <p>
                    Tu asistente virtual de apoyo emocional está aquí para ayudarte. 
                    Comparte tus sentimientos y encuentra apoyo cuando lo necesites.
                </p>
                <a href="aurora.php" class="btn">
                    <span>💬</span> Iniciar conversación
                </a>
                
                <?php if ($stats['conversaciones'] > 0): ?>
                <div class="stats-grid">
                    <div class="stat-box">
                        <span class="stat-value"><?php echo $stats['conversaciones']; ?></span>
                        <span class="stat-label">Conversaciones</span>
                    </div>
                </div>
                <?php endif; ?>
            </div>
            
            <!-- Card de Foto de Perfil (Solo Premium) -->
            <?php if ($_SESSION['usuario_plan'] === 'premium'): ?>
            <div class="card">
                <div class="card-header">
                    <span class="card-icon">📸</span>
                    <h3>Foto de perfil</h3>
                </div>
                <form method="POST" enctype="multipart/form-data" id="uploadForm">
                    <label for="foto_input" class="upload-area">
                        <div class="upload-icon">📷</div>
                        <div class="upload-text">Haz clic para subir una foto</div>
                        <div class="upload-hint">
                            JPG, PNG, GIF o WEBP (máximo 5MB)
                        </div>
                        <input type="file" 
                               name="foto_perfil" 
                               id="foto_input" 
                               accept="image/jpeg,image/png,image/gif,image/webp" 
                               onchange="handleFileSelect(this)">
                    </label>
                </form>
            </div>
            <?php endif; ?>
            
            <!-- Card de Configuración -->
            <div class="card">
                <div class="card-header">
                    <span class="card-icon">⚙️</span>
                    <h3>Configuración</h3>
                </div>
                <ul class="feature-list">
                    <li>
                        <a href="cambiar_password.php">Cambiar contraseña</a>
                    </li>
                    <li>Email: <?php echo htmlspecialchars($_SESSION['usuario_email']); ?></li>
                    <li>Plan: <?php echo ucfirst($_SESSION['usuario_plan']); ?></li>
                    <li>Miembro desde: <?php 
                        $fecha_registro = new DateTime($_SESSION['usuario_fecha_registro'] ?? 'now');
                        echo $fecha_registro->format('d/m/Y'); 
                    ?></li>
                </ul>
            </div>
            
            <!-- Card Premium (Solo para usuarios gratuitos) -->
            <?php if ($_SESSION['usuario_plan'] === 'gratuito'): ?>
            <div class="card premium-card">
                <div class="card-header">
                    <span class="card-icon">✨</span>
                    <h3>Mejora a Premium</h3>
                </div>
                <p>
                    Desbloquea todas las funciones de Aurora y lleva tu experiencia al siguiente nivel por solo <strong>S/ 100</strong>
                </p>
                <ul class="feature-list">
                    <li>Conversaciones ilimitadas</li>
                    <li>Foto de perfil personalizada</li>
                    <li>Respuestas prioritarias</li>
                    <li>Análisis emocional avanzado</li>
                    <li>Soporte prioritario 24/7</li>
                    <li>Sin anuncios</li>
                </ul>
                <a href="upgrade.php" class="btn">
                    <span>🚀</span> Actualizar ahora
                </a>
            </div>
            <?php endif; ?>
        </div>
    </div>

    <script>
       // Manejo de selección de archivo con preview
function handleFileSelect(input) {
    if (input.files && input.files[0]) {
        const file = input.files[0];
        const maxSize = 5 * 1024 * 1024; // 5MB
        
        // Validar tamaño del archivo
        if (file.size > maxSize) {
            alert("El archivo es demasiado grande. Máximo permitido: 5MB");
            input.value = ""; // Limpiar el input file
            return;
        }

        // Leer el archivo seleccionado
        const reader = new FileReader();
        reader.onload = (e) => {
            const imageDataUrl = e.target.result;
            // Mostrar preview de la imagen
            const uploadArea = document.querySelector(".upload-area");
            const uploadIcon = uploadArea.querySelector(".upload-icon");
            const uploadText = uploadArea.querySelector(".upload-text");
            const uploadHint = uploadArea.querySelector(".upload-hint");

            // Ocultar elementos innecesarios y mostrar la imagen
            uploadIcon.style.display = "none";
            uploadText.style.display = "none";
            uploadHint.style.display = "none";
            uploadArea.style.backgroundImage = `url(${imageDataUrl})`;
            uploadArea.style.backgroundSize = "cover";
            uploadArea.style.backgroundPosition = "center";
        };
        reader.readAsDataURL(file);
    }
}
    </script>
</body>     
</html>