<?php
require_once '../CBDD/config.php';

// Verificar si el usuario está logueado
if (!isset($_SESSION['usuario_id'])) {
    header('Location: ../index.php');
    exit();
}

// Procesar subida de foto
if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_FILES['foto_perfil'])) {
    // Solo Premium puede subir foto
    if ($_SESSION['usuario_plan'] !== 'premium') {
        $mensaje = 'Solo usuarios Premium pueden subir foto de perfil';
        $tipo_mensaje = 'error';
    } else {
        $file = $_FILES['foto_perfil'];

        if ($file['error'] === 0) {
            $allowedExt = ['jpg', 'jpeg', 'png', 'gif'];
            $filename = $file['name'];
            $file_ext = strtolower(pathinfo($filename, PATHINFO_EXTENSION));

            // Validar formato
            if (!in_array($file_ext, $allowedExt)) {
                $mensaje = 'Formato de archivo no permitido (solo JPG, PNG, GIF)';
                $tipo_mensaje = 'error';
            } else {
                // Crear carpeta si no existe
                if (!file_exists(UPLOAD_DIR)) {
                    mkdir(UPLOAD_DIR, 0777, true);
                }

                // Nombre único
                $new_filename = 'user_' . $_SESSION['usuario_id'] . '_' . time() . '.' . $file_ext;
                $destination = UPLOAD_DIR . $new_filename;

                // Mover archivo
                if (move_uploaded_file($file['tmp_name'], $destination)) {
                    try {
                        $conn = getConnection();
                        $stmt = $conn->prepare("UPDATE usuarios SET foto_perfil = ? WHERE id = ?");
                        $stmt->execute([$new_filename, $_SESSION['usuario_id']]);

                        $_SESSION['usuario_foto'] = $new_filename;

                        $mensaje = 'Foto de perfil actualizada exitosamente';
                        $tipo_mensaje = 'exito';
                    } catch (PDOException $e) {
                        $mensaje = 'Error al actualizar foto: ' . $e->getMessage();
                        $tipo_mensaje = 'error';
                    }
                } else {
                    $mensaje = 'Error al subir el archivo';
                    $tipo_mensaje = 'error';
                }
            }
        } else {
            $mensaje = 'No se recibió ningún archivo o hubo un error al subirlo';
            $tipo_mensaje = 'error';
        }
    }
}
?>
<!DOCTYPE html>
<html lang="es">
<head>
  <meta charset="UTF-8">
  <title>Aurora - Tu compañera emocional</title>
  <meta name="viewport" content="width=device-width, initial-scale=1.0">
  <link rel="stylesheet" href="../CSS/aurora.css">
</head>
<body>
  
  <!-- Barra de navegación superior -->
  <div class="top-navbar">
    <div class="navbar-left">
      <button id="btnVolver" onclick="goBack()" title="Volver">
        ← Volver
      </button>
      <button id="btnInicio" onclick="goHome()" title="Inicio">
        🏠 Inicio
      </button>
    </div>
    
    <div class="navbar-center">
      <h2>Aurora - Asistente Emocional</h2>
    </div>
    
    <div class="navbar-right">
      <button id="profileBtn" onclick="toggleProfile()" title="Perfil">
        <img id="userProfilePic" src="<?php echo !empty($_SESSION['usuario_foto']) ? 'uploads/' . htmlspecialchars($_SESSION['usuario_foto']) : 'https://via.placeholder.com/40'; ?>" alt="Perfil">
        <span id="userName"><?php echo htmlspecialchars($_SESSION['usuario_nombre'] ?? 'Usuario'); ?></span>
      </button>
      
      <button id="menuBtn" onclick="toggleMenu()" title="Menú">
        ☰
      </button>
    </div>
  </div>

  <!-- Menú desplegable -->
  <div id="menuDropdown" class="menu-dropdown hidden">
    <button onclick="openModal('settingsModal')">⚙️ Configuración</button>
    <button onclick="openModal('historyModal')">📜 Historial</button>
    <button onclick="openModal('statsModal')">📊 Estadísticas</button>
    <button onclick="logout()">🚪 Cerrar sesión</button>
  </div>

  <!-- Panel de perfil (sidebar) -->
  <div id="profilePanel" class="profile-panel hidden">
    <div class="profile-header">
      <h3>Mi Perfil</h3>
      <button onclick="toggleProfile()" class="close-btn">✕</button>
    </div>
    
    <div class="profile-content">
      <div class="profile-pic-container">
        <img id="profilePicLarge" src="<?php echo !empty($_SESSION['usuario_foto']) ? 'uploads/' . htmlspecialchars($_SESSION['usuario_foto']) : 'https://via.placeholder.com/150'; ?>" alt="Foto de perfil">
        
        <?php if ($_SESSION['usuario_plan'] === 'premium'): ?>
        <form method="POST" enctype="multipart/form-data" id="photoUploadForm">
          <label for="uploadPhoto" class="upload-photo-btn">
            📷 Cambiar foto
          </label>
          <input type="file" name="foto_perfil" id="uploadPhoto" accept="image/*" onchange="document.getElementById('photoUploadForm').submit()" style="display: none;">
        </form>
        <?php else: ?>
        <label class="upload-photo-btn disabled">
          📷 Cambiar foto
        </label>
        <p class="premium-note">⭐ Función Premium</p>
        <?php endif; ?>
      </div>
      
      <div class="profile-info">
        <p><strong>Nombre:</strong> <span id="profileName"><?php echo htmlspecialchars($_SESSION['usuario_nombre'] ?? 'Usuario'); ?></span></p>
        <p><strong>Email:</strong> <span id="profileEmail"><?php echo htmlspecialchars($_SESSION['usuario_email'] ?? 'email@ejemplo.com'); ?></span></p>
        <p><strong>Plan:</strong> <span id="profilePlan" class="plan-badge"><?php echo strtoupper($_SESSION['usuario_plan'] ?? 'free'); ?></span></p>
        <p><strong>Miembro desde:</strong> <span id="memberSince"><?php echo date('d/m/Y'); ?></span></p>
      </div>
      
      <div class="profile-stats">
        <h4>Estadísticas de uso</h4>
        <div class="stat-item">
          <span>🗣️ Conversaciones:</span>
          <span id="totalConversations">0</span>
        </div>
        <div class="stat-item">
          <span>⏱️ Tiempo total:</span>
          <span id="totalTime">0 min</span>
        </div>
        <div class="stat-item">
          <span>⭐ Calificación promedio:</span>
          <span id="avgRating">0.0</span>
        </div>
      </div>
      
      <?php if ($_SESSION['usuario_plan'] !== 'premium'): ?>
      <button class="upgrade-btn" onclick="upgradeToPremium()">
        ⭐ Actualizar a Premium
      </button>
      <?php endif; ?>
    </div>
  </div>

  <!-- Panel de análisis emocional -->
  <div class="emotion-panel">
    <div class="emotion-header">
      📊 Análisis Emocional en Tiempo Real
      <button id="toggleEmotionPanel" onclick="toggleEmotionPanel()">▼</button>
    </div>
    
    <div id="emotionPanelContent" class="emotion-panel-content">
      <div id="videoContainer">
        <video id="webcam" autoplay playsinline></video>
        <canvas id="canvas"></canvas>
        <button id="toggleCameraBtn" onclick="toggleCamera()">📷 Pausar cámara</button>
      </div>

      <div class="emotion-stats">
        <div id="emotionBars">
          <div class="emotion-bar">
            <div class="emotion-bar-label">
              <span>😊 Feliz</span>
              <span id="happy-val">0%</span>
            </div>
            <div class="emotion-bar-fill">
              <div id="happy-bar" class="emotion-bar-value" style="width: 0%"></div>
            </div>
          </div>

          <div class="emotion-bar">
            <div class="emotion-bar-label">
              <span>😢 Triste</span>
              <span id="sad-val">0%</span>
            </div>
            <div class="emotion-bar-fill">
              <div id="sad-bar" class="emotion-bar-value" style="width: 0%"></div>
            </div>
          </div>

          <div class="emotion-bar">
            <div class="emotion-bar-label">
              <span>😠 Enojado</span>
              <span id="angry-val">0%</span>
            </div>
            <div class="emotion-bar-fill">
              <div id="angry-bar" class="emotion-bar-value" style="width: 0%"></div>
            </div>
          </div>

          <div class="emotion-bar">
            <div class="emotion-bar-label">
              <span>😐 Neutral</span>
              <span id="neutral-val">0%</span>
            </div>
            <div class="emotion-bar-fill">
              <div id="neutral-bar" class="emotion-bar-value" style="width: 0%"></div>
            </div>
          </div>
        </div>

        <div style="margin-top: 12px; padding-top: 12px; border-top: 1px solid #ddd;">
          <strong>Estado actual:</strong>
          <div id="currentEmotion" style="color: #667eea; font-size: 16px; margin-top: 5px;">
            Esperando análisis...
          </div>
        </div>
        
        <button id="saveEmotionBtn" onclick="saveEmotionalSnapshot()">
          💾 Guardar snapshot emocional
        </button>
      </div>
    </div>
  </div>

  <!-- Sección de chat -->
  <div class="chat-section">
    <div class="chat-overlay">
      <!-- Controles de chat -->
      <div class="chat-controls-top">
        <button onclick="clearChat()" title="Limpiar chat">🗑️</button>
        <button onclick="exportChat()" title="Exportar conversación">📥</button>
        <button onclick="toggleTranscription()" title="Ver transcripción">📝</button>
        <button onclick="pauseConversation()" title="Pausar conversación">⏸️</button>
      </div>

      <div class="aurora-avatar">
        <div id="avatarContainer">
          <img id="avatarImg" src="https://images.unsplash.com/photo-1438761681033-6461ffad8d80?q=80&w=400" 
               alt="Aurora" class="avatar-img">
          <div class="sound-wave" id="soundWave1"></div>
          <div class="sound-wave" id="soundWave2" style="animation-delay: 0.5s;"></div>
          <div class="sound-wave" id="soundWave3" style="animation-delay: 1s;"></div>
          <div class="volume-indicator" id="volumeIndicator">
            <div class="volume-bar"></div>
            <div class="volume-bar"></div>
            <div class="volume-bar"></div>
            <div class="volume-bar"></div>
            <div class="volume-bar"></div>
            <div class="volume-bar"></div>
          </div>
        </div>
      </div>

      <div class="chat-title">Aurora 🌸</div>

      <div class="chat-container" id="chat">
        <div class="bubble aurora">Hola, soy Aurora. Estoy aquí para escucharte siempre. ¿Cómo te sientes hoy?</div>
      </div>

      <!-- Panel de transcripción -->
      <div id="transcriptionPanel" class="transcription-panel hidden">
        <h4>Transcripción de la conversación</h4>
        <div id="transcriptionContent"></div>
        <button onclick="downloadTranscription()">💾 Descargar</button>
      </div>

      <div class="controls">
        <div class="voice-selector" style="display: none;">
          <label for="voiceSelect">🎤 Voz:</label>
          <select id="voiceSelect"></select>
        </div>

        <div class="input-controls">
          <button id="micBtn" onclick="toggleListening()">
            🎙️ Presiona para hablar
          </button>
          
          <input type="text" id="textInput" placeholder="O escribe aquí..." onkeypress="handleTextInput(event)">
          <button id="sendBtn" onclick="sendTextMessage()">📤</button>
        </div>

        <div class="status-indicator" id="status"></div>

        <!-- Sugerencias rápidas -->
        <div class="quick-suggestions">
          <button onclick="sendQuickMessage('¿Cómo estás?')">👋 Saludar</button>
          <button onclick="sendQuickMessage('Me siento ansioso')">😰 Ansiedad</button>
          <button onclick="sendQuickMessage('Necesito un consejo')">💡 Consejo</button>
          <button onclick="sendQuickMessage('Cuéntame algo positivo')">✨ Positivo</button>
        </div>

        <!-- Sistema de calificación -->
        <div class="rating-container">
          <h3>¿Cómo fue mi ayuda?</h3>
          <div class="stars" id="starsContainer">
            <span class="star" data-value="1" onclick="rateAurora(1)">⭐</span>
            <span class="star" data-value="2" onclick="rateAurora(2)">⭐</span>
            <span class="star" data-value="3" onclick="rateAurora(3)">⭐</span>
            <span class="star" data-value="4" onclick="rateAurora(4)">⭐</span>
            <span class="star" data-value="5" onclick="rateAurora(5)">⭐</span>
          </div>
          <div class="rating-feedback" id="ratingFeedback"></div>
        </div>
      </div>
    </div>
  </div>

  <!-- Modal de configuración -->
  <div id="settingsModal" class="modal hidden">
    <div class="modal-content">
      <span class="close-modal" onclick="closeModal('settingsModal')">&times;</span>
      <h2>⚙️ Configuración</h2>
      
      <div class="settings-section">
        <h3>Audio</h3>
        <label>
          Volumen de voz:
          <input type="range" id="volumeRange" min="0" max="1" step="0.1" value="1" onchange="updateVolume(this.value)">
          <span id="volumeValue">100%</span>
        </label>
        
        <label>
          Velocidad de habla:
          <input type="range" id="rateRange" min="0.5" max="2" step="0.1" value="0.93" onchange="updateRate(this.value)">
          <span id="rateValue">0.93x</span>
        </label>
        
        <label>
          <input type="checkbox" id="autoListenCheck" checked onchange="toggleAutoListen(this.checked)">
          Activar escucha automática
        </label>
      </div>
      
      <div class="settings-section">
        <h3>Privacidad</h3>
        <label>
          <input type="checkbox" id="saveChatCheck" checked onchange="toggleSaveChat(this.checked)">
          Guardar conversaciones
        </label>
        
        <label>
          <input type="checkbox" id="saveEmotionsCheck" checked onchange="toggleSaveEmotions(this.checked)">
          Guardar datos emocionales
        </label>
        
        <button onclick="deleteAllData()">🗑️ Eliminar todos mis datos</button>
      </div>
      
      <div class="settings-section">
        <h3>Notificaciones</h3>
        <label>
          <input type="checkbox" id="notifCheck" onchange="toggleNotifications(this.checked)">
          Permitir notificaciones
        </label>
      </div>
    </div>
  </div>

  <!-- Modal de historial -->
  <div id="historyModal" class="modal hidden">
    <div class="modal-content">
      <span class="close-modal" onclick="closeModal('historyModal')">&times;</span>
      <h2>📜 Historial de Conversaciones</h2>
      <div id="historyContent">
        <p>Cargando historial...</p>
      </div>
    </div>
  </div>

  <!-- Modal de estadísticas -->
  <div id="statsModal" class="modal hidden">
    <div class="modal-content">
      <span class="close-modal" onclick="closeModal('statsModal')">&times;</span>
      <h2>📊 Estadísticas Emocionales</h2>
      <div id="statsContent">
        <canvas id="emotionChart"></canvas>
      </div>
    </div>
  </div>

  <!-- Face-api.js para detección de emociones -->
  <script defer src="https://cdn.jsdelivr.net/npm/face-api.js@0.22.2/dist/face-api.min.js"></script>
  <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>

  <script>
    // Variables globales
    const chat = document.getElementById('chat');
    const synth = window.speechSynthesis;
    const micBtn = document.getElementById('micBtn');
    const statusDiv = document.getElementById('status');
    const avatarImg = document.getElementById('avatarImg');
    const soundWave1 = document.getElementById('soundWave1');
    const soundWave2 = document.getElementById('soundWave2');
    const soundWave3 = document.getElementById('soundWave3');
    const volumeIndicator = document.getElementById('volumeIndicator');
    const starsContainer = document.getElementById('starsContainer');
    const ratingFeedback = document.getElementById('ratingFeedback');
    
    let recognition;
    let voices = [];
    let isListening = false;
    let emotionInterval;
    let isSpeaking = false;
    let currentRating = 0;
    let conversationId = Date.now();
    let cameraActive = true;
    let autoListen = true;
    let conversationPaused = false;
    let settings = {
      volume: 1,
      rate: 0.93,
      pitch: 1.02,
      saveChat: true,
      saveEmotions: true
    };

    // ========================================
    // FUNCIONES DE NAVEGACIÓN
    // ========================================
    function goBack() {
      window.history.back();
    }

    function goHome() {
      window.location.href = 'dashboard.php';
    }

    function logout() {
      if (confirm('¿Estás seguro que deseas cerrar sesión?')) {
        window.location.href = '../BACK/logout.php';
      }
    }

    // ========================================
    // FUNCIONES DE UI
    // ========================================
    function toggleProfile() {
      const panel = document.getElementById('profilePanel');
      panel.classList.toggle('hidden');
    }

    function toggleMenu() {
      const menu = document.getElementById('menuDropdown');
      menu.classList.toggle('hidden');
    }

    function openModal(modalId) {
      const modal = document.getElementById(modalId);
      modal.classList.remove('hidden');
      
      if (modalId === 'historyModal') loadHistory();
      if (modalId === 'statsModal') loadStats();
    }

    function closeModal(modalId) {
      const modal = document.getElementById(modalId);
      modal.classList.add('hidden');
    }

    function toggleEmotionPanel() {
      const content = document.getElementById('emotionPanelContent');
      const btn = document.getElementById('toggleEmotionPanel');
      content.classList.toggle('hidden');
      btn.textContent = content.classList.contains('hidden') ? '▼' : '▲';
    }

    function toggleTranscription() {
      const panel = document.getElementById('transcriptionPanel');
      panel.classList.toggle('hidden');
      updateTranscription();
    }

    function updateTranscription() {
      const content = document.getElementById('transcriptionContent');
      const bubbles = chat.querySelectorAll('.bubble');
      let transcript = '';
      
      bubbles.forEach(bubble => {
        const sender = bubble.classList.contains('aurora') ? 'Aurora' : 'Usuario';
        transcript += `${sender}: ${bubble.textContent}\n\n`;
      });
      
      content.textContent = transcript || 'No hay conversación aún.';
    }

    function downloadTranscription() {
      const content = document.getElementById('transcriptionContent').textContent;
      const blob = new Blob([content], { type: 'text/plain' });
      const url = URL.createObjectURL(blob);
      const a = document.createElement('a');
      a.href = url;
      a.download = `aurora_conversacion_${new Date().toISOString().split('T')[0]}.txt`;
      a.click();
      URL.revokeObjectURL(url);
    }

    // ========================================
    // SISTEMA DE CALIFICACIÓN
    // ========================================
    function rateAurora(rating) {
      currentRating = rating;
      
      const stars = starsContainer.querySelectorAll('.star');
      stars.forEach((star, index) => {
        if (index < rating) {
          star.classList.add('active');
        } else {
          star.classList.remove('active');
        }
      });

      const messages = {
        1: '😢 Lamento no haber sido de ayuda',
        2: '😊 Trabajaré para mejorar',
        3: '🙂 Gracias por tu opinión',
        4: '😄 ¡Me alegra haberte ayudado!',
        5: '🌟 ¡Gracias! Siempre estaré aquí para ti'
      };

      ratingFeedback.textContent = messages[rating];
      ratingFeedback.classList.add('show');

      sendRatingToServer(rating);

      setTimeout(() => {
        ratingFeedback.classList.remove('show');
      }, 3000);
    }

    function sendRatingToServer(rating) {
      fetch('../BACK/calificacion.php', {
        method: 'POST',
        headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
        body: `rating=${rating}&conversation_id=${conversationId}&timestamp=${new Date().toISOString()}`
      })
      .then(res => res.json())
      .then(data => {
        if (data.success) {
          console.log('Calificación guardada exitosamente');
        }
      })
      .catch(err => console.error('Error al enviar calificación:', err));
    }

    // Efecto hover en estrellas
    starsContainer.addEventListener('mouseover', (e) => {
      if (e.target.classList.contains('star')) {
        const value = parseInt(e.target.dataset.value);
        const stars = starsContainer.querySelectorAll('.star');
        stars.forEach((star, index) => {
          if (index < value) {
            star.style.filter = 'grayscale(0)';
            star.style.opacity = '1';
          } else {
            star.style.filter = 'grayscale(1)';
            star.style.opacity = '0.3';
          }
        });
      }
    });

    starsContainer.addEventListener('mouseout', () => {
      const stars = starsContainer.querySelectorAll('.star');
      stars.forEach((star, index) => {
        if (index < currentRating) {
          star.classList.add('active');
        } else {
          star.classList.remove('active');
          star.style.filter = 'grayscale(1)';
          star.style.opacity = '0.3';
        }
      });
    });

    // ========================================
    // INICIALIZACIÓN DE VOCES
    // ========================================
    function loadVoices() {
      voices = synth.getVoices();
      const voiceSelect = document.getElementById('voiceSelect');
      voiceSelect.innerHTML = '';

      const googleSpanishVoices = voices.filter(v => 
        v.lang.startsWith('es') && v.name.toLowerCase().includes('google')
      );

      const esFallbackVoices = voices.filter(v => v.lang.startsWith('es'));
      const finalVoices = googleSpanishVoices.length > 0 ? googleSpanishVoices : esFallbackVoices;

      finalVoices.forEach((voice, i) => {
        const option = document.createElement('option');
        option.value = i;
        option.textContent = voice.name;
        voiceSelect.appendChild(option);
      });
    }

    if (synth.onvoiceschanged !== undefined) {
      synth.onvoiceschanged = loadVoices;
    }
    loadVoices();

    // ========================================
    // SÍNTESIS DE VOZ
    // ========================================
    function speak(text) {
      if (!text || conversationPaused) return;
      
      isSpeaking = true;
      const utterance = new SpeechSynthesisUtterance(text);

      const googleVoice = voices.find(v =>
        v.lang.startsWith("es") && v.name.toLowerCase().includes("google")
      ) || voices.find(v => v.lang.startsWith("es"));

      utterance.voice = googleVoice;
      utterance.lang = "es-ES";
      utterance.pitch = settings.pitch;
      utterance.rate = settings.rate;
      utterance.volume = settings.volume;

      utterance.text = text
        .replace(/,/g, ", ")
        .replace(/\./g, ". ")
        .replace(/\?/g, "? ")
        .replace(/\!/g, "! ");

      avatarImg.classList.add('talking');
      soundWave1.classList.add('active');
      soundWave2.classList.add('active');
      soundWave3.classList.add('active');
      volumeIndicator.classList.add('active');

      if (isListening && recognition) {
        try { recognition.stop(); } catch(e) {}
        isListening = false;
      }

      utterance.onend = () => {
        isSpeaking = false;
        avatarImg.classList.remove('talking');
        soundWave1.classList.remove('active');
        soundWave2.classList.remove('active');
        soundWave3.classList.remove('active');
        volumeIndicator.classList.remove('active');

        if (autoListen && !conversationPaused) {
          setTimeout(() => {
            if (!isListening && !isSpeaking) startListening();
          }, 600);
        }
      };

      synth.speak(utterance);
    }

    // ========================================
    // AGREGAR BURBUJAS DE CHAT
    // ========================================
    function addBubble(text, sender = 'aurora') {
      const div = document.createElement('div');
      div.className = `bubble ${sender}`;
      div.textContent = text;
      chat.appendChild(div);
      chat.scrollTop = chat.scrollHeight;
      
      if (settings.saveChat) {
        saveToLocalStorage();
      }
    }

    function saveToLocalStorage() {
      const bubbles = Array.from(chat.querySelectorAll('.bubble')).map(b => ({
        text: b.textContent,
        sender: b.classList.contains('aurora') ? 'aurora' : 'user'
      }));
      localStorage.setItem('aurora_chat', JSON.stringify(bubbles));
    }

    function loadFromLocalStorage() {
      const saved = localStorage.getItem('aurora_chat');
      if (saved && settings.saveChat) {
        const bubbles = JSON.parse(saved);
        chat.innerHTML = '';
        bubbles.forEach(b => addBubble(b.text, b.sender));
      }
    }

    // ========================================
    // RECONOCIMIENTO DE VOZ
    // ========================================
    function toggleListening() {
      if (!('webkitSpeechRecognition' in window)) {
        alert('Tu navegador no soporta reconocimiento de voz. Usa Chrome o Edge.');
        return;
      }

      if (isListening) {
        stopListening();
      } else {
        startListening();
      }
    }

    function startListening() {
      if (synth.speaking || conversationPaused) {
        synth.cancel();
        return;
      }

      recognition = new webkitSpeechRecognition();
      recognition.lang = 'es-ES';
      recognition.continuous = true;
      recognition.interimResults = false;
      recognition.maxAlternatives = 1;

      recognition.onstart = () => {
        isListening = true;
        micBtn.classList.add('listening');
        micBtn.textContent = '🔴 Escuchando...';
        statusDiv.textContent = 'Te estoy escuchando siempre 💭';
      };

      recognition.onresult = (event) => {
        const transcript = event.results[event.results.length - 1][0].transcript.trim();
        if (transcript.length > 0) {
          addBubble(transcript, 'user');
          sendToAurora(transcript);
        }
      };

      recognition.onerror = (e) => {
        console.error('Error de reconocimiento:', e.error);
        if (e.error === 'no-speech') {
          setTimeout(() => {
            if (isListening && !conversationPaused) {
              try { recognition.start(); } catch (err) {}
            }
          }, 100);
        } else if (e.error !== 'aborted') {
          statusDiv.textContent = 'Error en el micrófono. Reintentando...';
          setTimeout(() => {
            if (isListening && !conversationPaused) {
              try { recognition.start(); } catch (err) {}
            }
          }, 1000);
        }
      };

      recognition.onend = () => {
        if (isListening && !isSpeaking && !conversationPaused) {
          setTimeout(() => {
            try { recognition.start(); } catch(e) {}
          }, 100);
        }
      };

      try {
        recognition.start();
      } catch (e) {
        console.error('Error al iniciar reconocimiento:', e);
      }
    }

    function stopListening() {
      if (recognition) {
        isListening = false;
        try {
          recognition.stop();
        } catch (e) {}
        micBtn.classList.remove('listening');
        micBtn.textContent = '🎙️ Presiona para hablar';
        statusDiv.textContent = '';
      }
    }

    // ========================================
    // ENVÍO AL API
    // ========================================
    function sendToAurora(text) {
      if (!text || conversationPaused) return;
      
      statusDiv.textContent = '🤔 Pensando...';
      micBtn.classList.add('processing');
      
      if (isListening) {
        try { recognition.stop(); } catch (e) {}
      }
      
      fetch('../BACK/987654321123456789_ia.php', {
        method: 'POST',
        headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
        body: `pregunta=${encodeURIComponent(text)}`
      })
      .then(res => res.json())
      .then(data => {
        statusDiv.textContent = '';
        micBtn.classList.remove('processing');
        
        if (data.success) {
          addBubble(data.respuesta, 'aurora');
          speak(data.respuesta);
        } else {
          const errorMsg = 'Disculpa, no pude procesar eso. ¿Podrías repetirlo?';
          addBubble(errorMsg, 'aurora');
          speak(errorMsg);
        }
      })
      .catch(err => {
        console.error('Error:', err);
        statusDiv.textContent = '';
        micBtn.classList.remove('processing');
        const errorMsg = 'Ocurrió un error de conexión. Estoy aquí para cuando quieras hablar.';
        addBubble(errorMsg, 'aurora');
        speak(errorMsg);
      });
    }

    // ========================================
    // FUNCIONES DE INPUT DE TEXTO
    // ========================================
    function handleTextInput(event) {
      if (event.key === 'Enter') {
        sendTextMessage();
      }
    }

    function sendTextMessage() {
      const input = document.getElementById('textInput');
      const text = input.value.trim();
      if (text) {
        addBubble(text, 'user');
        sendToAurora(text);
        input.value = '';
      }
    }

    function sendQuickMessage(message) {
      addBubble(message, 'user');
      sendToAurora(message);
    }

    // ========================================
    // CONTROLES DE CONVERSACIÓN
    // ========================================
    function clearChat() {
      if (confirm('¿Estás seguro que deseas limpiar el chat?')) {
        chat.innerHTML = '<div class="bubble aurora">Chat limpiado. ¿En qué puedo ayudarte?</div>';
        localStorage.removeItem('aurora_chat');
        conversationId = Date.now();
      }
    }

    function exportChat() {
      const bubbles = chat.querySelectorAll('.bubble');
      let content = 'Conversación con Aurora\n';
      content += `Fecha: ${new Date().toLocaleString()}\n\n`;
      
      bubbles.forEach(bubble => {
        const sender = bubble.classList.contains('aurora') ? 'Aurora' : 'Usuario';
        content += `${sender}: ${bubble.textContent}\n\n`;
      });
      
      const blob = new Blob([content], { type: 'text/plain' });
      const url = URL.createObjectURL(blob);
      const a = document.createElement('a');
      a.href = url;
      a.download = `aurora_${new Date().toISOString().split('T')[0]}.txt`;
      a.click();
      URL.revokeObjectURL(url);
    }

    function pauseConversation() {
      conversationPaused = !conversationPaused;
      
      if (conversationPaused) {
        stopListening();
        synth.cancel();
        statusDiv.textContent = '⏸️ Conversación pausada';
      } else {
        statusDiv.textContent = '';
        if (autoListen) startListening();
      }
    }

    // ========================================
    // DETECCIÓN DE EMOCIONES
    // ========================================
    const video = document.getElementById('webcam');
    const canvas = document.getElementById('canvas');

    async function initFaceDetection() {
      try {
        await faceapi.nets.tinyFaceDetector.loadFromUri('https://cdn.jsdelivr.net/npm/@vladmandic/face-api/model');
        await faceapi.nets.faceExpressionNet.loadFromUri('https://cdn.jsdelivr.net/npm/@vladmandic/face-api/model');

        const stream = await navigator.mediaDevices.getUserMedia({ 
          video: { 
            width: { ideal: 640 },
            height: { ideal: 480 }
          } 
        });
        video.srcObject = stream;

        video.addEventListener('loadeddata', () => {
          canvas.width = video.videoWidth;
          canvas.height = video.videoHeight;
          detectEmotions();
        });
      } catch (err) {
        console.error('Error al iniciar cámara:', err);
        document.getElementById('currentEmotion').textContent = 'Cámara no disponible 📷';
      }
    }

    async function detectEmotions() {
      emotionInterval = setInterval(async () => {
        if (!cameraActive) return;
        
        const detections = await faceapi.detectSingleFace(
          video,
          new faceapi.TinyFaceDetectorOptions()
        ).withFaceExpressions();

        if (detections) {
          const expressions = detections.expressions;
          updateEmotionBars(expressions);
          
          const dominantEmotion = Object.keys(expressions).reduce((a, b) =>
            expressions[a] > expressions[b] ? a : b
          );
          
          const emotionLabels = {
            happy: '😊 Feliz',
            sad: '😢 Triste',
            angry: '😠 Enojado',
            neutral: '😐 Neutral',
            surprised: '😲 Sorprendido',
            disgusted: '🤢 Disgustado',
            fearful: '😨 Asustado'
          };

          document.getElementById('currentEmotion').textContent = 
            emotionLabels[dominantEmotion] || 'Analizando...';
        } else {
          document.getElementById('currentEmotion').textContent = 'No se detecta rostro';
        }
      }, 1000);
    }

    function updateEmotionBars(expressions) {
      const emotions = ['happy', 'sad', 'angry', 'neutral'];
      
      emotions.forEach(emotion => {
        const percentage = Math.round(expressions[emotion] * 100);
        const bar = document.getElementById(`${emotion}-bar`);
        const val = document.getElementById(`${emotion}-val`);
        
        if (bar && val) {
          bar.style.width = `${percentage}%`;
          val.textContent = `${percentage}%`;
        }
      });
    }

    function toggleCamera() {
      cameraActive = !cameraActive;
      const btn = document.getElementById('toggleCameraBtn');
      
      if (cameraActive) {
        btn.textContent = '📷 Pausar cámara';
        video.play();
      } else {
        btn.textContent = '📷 Reanudar cámara';
        video.pause();
      }
    }

    function saveEmotionalSnapshot() {
      const emotions = {
        happy: document.getElementById('happy-val').textContent,
        sad: document.getElementById('sad-val').textContent,
        angry: document.getElementById('angry-val').textContent,
        neutral: document.getElementById('neutral-val').textContent,
        current: document.getElementById('currentEmotion').textContent,
        timestamp: new Date().toISOString()
      };
      
      if (settings.saveEmotions) {
        fetch('../BACK/guardar_emocion.php', {
          method: 'POST',
          headers: { 'Content-Type': 'application/json' },
          body: JSON.stringify(emotions)
        })
        .then(res => res.json())
        .then(data => {
          if (data.success) {
            alert('Snapshot emocional guardado ✅');
          }
        })
        .catch(err => console.error('Error:', err));
      }
    }

    // ========================================
    // CONFIGURACIÓN
    // ========================================
    function updateVolume(value) {
      settings.volume = parseFloat(value);
      document.getElementById('volumeValue').textContent = Math.round(value * 100) + '%';
    }

    function updateRate(value) {
      settings.rate = parseFloat(value);
      document.getElementById('rateValue').textContent = value + 'x';
    }

    function toggleAutoListen(checked) {
      autoListen = checked;
      if (!checked && isListening) {
        stopListening();
      } else if (checked && !isListening && !isSpeaking) {
        startListening();
      }
    }

    function toggleSaveChat(checked) {
      settings.saveChat = checked;
      if (!checked) {
        localStorage.removeItem('aurora_chat');
      }
    }

    function toggleSaveEmotions(checked) {
      settings.saveEmotions = checked;
    }

    function toggleNotifications(checked) {
      if (checked && 'Notification' in window) {
        Notification.requestPermission();
      }
    }

    function deleteAllData() {
      if (confirm('¿Estás seguro? Esto eliminará todos tus datos guardados.')) {
        localStorage.clear();
        location.reload();
      }
    }

    function upgradeToPremium() {
      window.location.href = 'planes.php';
    }

    // ========================================
    // HISTORIAL Y ESTADÍSTICAS
    // ========================================
    function loadHistory() {
      const content = document.getElementById('historyContent');
      content.innerHTML = '<p>Cargando...</p>';
      
      fetch('../BACK/obtener_historial.php')
        .then(res => res.json())
        .then(data => {
          if (data.success && data.conversaciones.length > 0) {
            content.innerHTML = data.conversaciones.map(c => `
              <div class="history-item">
                <h4>${new Date(c.fecha).toLocaleDateString()}</h4>
                <p>${c.resumen}</p>
              </div>
            `).join('');
          } else {
            content.innerHTML = '<p>No hay historial disponible.</p>';
          }
        })
        .catch(err => {
          content.innerHTML = '<p>Error al cargar historial.</p>';
        });
    }

    function loadStats() {
      // Aquí se cargarían las estadísticas desde el servidor
      // Por ahora mostraremos un placeholder
      const ctx = document.getElementById('emotionChart');
      if (ctx) {
        new Chart(ctx, {
          type: 'bar',
          data: {
            labels: ['Feliz', 'Triste', 'Enojado', 'Neutral'],
            datasets: [{
              label: 'Emociones registradas',
              data: [12, 5, 3, 8],
              backgroundColor: ['#4CAF50', '#2196F3', '#F44336', '#9E9E9E']
            }]
          }
        });
      }
    }

    // ========================================
    // INICIALIZACIÓN
    // ========================================
    window.addEventListener('load', () => {
      loadFromLocalStorage();
      initFaceDetection();
      
      setTimeout(() => {
        const welcomeMsg = 'Hola, soy Aurora. Estoy aquí para escucharte siempre. ¿Cómo te sientes hoy?';
        speak(welcomeMsg);
      }, 1000);
      
      setTimeout(() => {
        if (autoListen) startListening();
      }, 6000);
    });

    document.addEventListener('visibilitychange', () => {
      if (!document.hidden && !isListening && !synth.speaking && autoListen && !conversationPaused) {
        setTimeout(() => startListening(), 500);
      }
    });

    // Cerrar modales al hacer clic fuera
    window.addEventListener('click', (e) => {
      if (e.target.classList.contains('modal')) {
        e.target.classList.add('hidden');
      }
    });
  </script>
</body>
</html>