<?php
/**
 * SISTEMA DE CALIFICACIÓN - AURORA
 * Backend profesional para guardar y analizar calificaciones
 * 
 * @file calificacion.php
 * @author Tu nombre
 * @version 1.0
 */

// Configuración de headers para CORS y JSON
header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, GET, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

// Manejo de solicitudes OPTIONS para CORS
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

// ========================================
// CONFIGURACIÓN DE BASE DE DATOS
// ========================================

// Cambiar estos valores según tu configuración
define('DB_HOST', '172.233.111.182');
define('DB_NAME', 'tecnodes_aurora_db');
define('DB_USER', 'tecnodes_elker');
define('DB_PASS', 'Eperez12#.');
define('DB_CHARSET', 'utf8mb4');

// ========================================
// FUNCIONES DE BASE DE DATOS
// ========================================

/**
 * Crea la conexión a la base de datos con manejo de errores
 * @return PDO|null
 */
function getDBConnection() {
    try {
        $dsn = "mysql:host=" . DB_HOST . ";dbname=" . DB_NAME . ";charset=" . DB_CHARSET;
        $options = [
            PDO::ATTR_ERRMODE            => PDO::ERRMODE_EXCEPTION,
            PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
            PDO::ATTR_EMULATE_PREPARES   => false,
        ];
        
        $pdo = new PDO($dsn, DB_USER, DB_PASS, $options);
        return $pdo;
        
    } catch (PDOException $e) {
        error_log("Error de conexión a BD: " . $e->getMessage());
        return null;
    }
}

/**
 * Inicializa la tabla de calificaciones si no existe
 * @param PDO $pdo
 * @return bool
 */
function initializeDatabase($pdo) {
    try {
        $sql = "CREATE TABLE IF NOT EXISTS aurora_ratings (
            id INT AUTO_INCREMENT PRIMARY KEY,
            conversation_id BIGINT NOT NULL,
            rating INT NOT NULL CHECK (rating BETWEEN 1 AND 5),
            timestamp DATETIME NOT NULL,
            user_ip VARCHAR(45),
            user_agent VARCHAR(255),
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            INDEX idx_conversation (conversation_id),
            INDEX idx_rating (rating),
            INDEX idx_timestamp (timestamp)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
        
        $pdo->exec($sql);
        return true;
        
    } catch (PDOException $e) {
        error_log("Error al crear tabla: " . $e->getMessage());
        return false;
    }
}

/**
 * Guarda una calificación en la base de datos
 * @param PDO $pdo
 * @param int $conversationId
 * @param int $rating
 * @param string $timestamp
 * @return bool
 */
function saveRating($pdo, $conversationId, $rating, $timestamp) {
    try {
        // Validar datos
        if ($rating < 1 || $rating > 5) {
            throw new Exception("Calificación inválida");
        }
        
        // Obtener información del usuario
        $userIp = $_SERVER['REMOTE_ADDR'] ?? 'unknown';
        $userAgent = substr($_SERVER['HTTP_USER_AGENT'] ?? '', 0, 255);
        
        // Preparar consulta
        $sql = "INSERT INTO aurora_ratings 
                (conversation_id, rating, timestamp, user_ip, user_agent) 
                VALUES (:conversation_id, :rating, :timestamp, :user_ip, :user_agent)";
        
        $stmt = $pdo->prepare($sql);
        
        // Ejecutar con parámetros seguros
        $success = $stmt->execute([
            ':conversation_id' => $conversationId,
            ':rating' => $rating,
            ':timestamp' => $timestamp,
            ':user_ip' => $userIp,
            ':user_agent' => $userAgent
        ]);
        
        return $success;
        
    } catch (PDOException $e) {
        error_log("Error al guardar calificación: " . $e->getMessage());
        return false;
    } catch (Exception $e) {
        error_log("Error de validación: " . $e->getMessage());
        return false;
    }
}

/**
 * Obtiene estadísticas de calificaciones
 * @param PDO $pdo
 * @return array
 */
function getRatingStats($pdo) {
    try {
        $sql = "SELECT 
                    COUNT(*) as total_ratings,
                    AVG(rating) as average_rating,
                    SUM(CASE WHEN rating = 5 THEN 1 ELSE 0 END) as five_star,
                    SUM(CASE WHEN rating = 4 THEN 1 ELSE 0 END) as four_star,
                    SUM(CASE WHEN rating = 3 THEN 1 ELSE 0 END) as three_star,
                    SUM(CASE WHEN rating = 2 THEN 1 ELSE 0 END) as two_star,
                    SUM(CASE WHEN rating = 1 THEN 1 ELSE 0 END) as one_star,
                    DATE(created_at) as date
                FROM aurora_ratings
                WHERE created_at >= DATE_SUB(NOW(), INTERVAL 30 DAY)
                GROUP BY DATE(created_at)
                ORDER BY date DESC";
        
        $stmt = $pdo->query($sql);
        return $stmt->fetchAll();
        
    } catch (PDOException $e) {
        error_log("Error al obtener estadísticas: " . $e->getMessage());
        return [];
    }
}

// ========================================
// MANEJO DE SOLICITUDES
// ========================================

/**
 * Envía una respuesta JSON
 * @param bool $success
 * @param string $message
 * @param array $data
 */
function sendResponse($success, $message = '', $data = []) {
    $response = [
        'success' => $success,
        'message' => $message,
        'data' => $data,
        'timestamp' => date('Y-m-d H:i:s')
    ];
    
    echo json_encode($response, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
    exit();
}

/**
 * Valida y sanitiza los datos de entrada
 * @return array|null
 */
function validateInput() {
    // Verificar método POST
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        return null;
    }
    
    // Obtener datos
    $rating = filter_input(INPUT_POST, 'rating', FILTER_VALIDATE_INT);
    $conversationId = filter_input(INPUT_POST, 'conversation_id', FILTER_VALIDATE_INT);
    $timestamp = filter_input(INPUT_POST, 'timestamp', FILTER_SANITIZE_STRING);
    
    // Validar datos obligatorios
    if ($rating === false || $rating === null) {
        sendResponse(false, 'Calificación inválida o no proporcionada');
    }
    
    if ($conversationId === false || $conversationId === null) {
        sendResponse(false, 'ID de conversación inválido o no proporcionado');
    }
    
    // Validar rango de calificación
    if ($rating < 1 || $rating > 5) {
        sendResponse(false, 'La calificación debe estar entre 1 y 5');
    }
    
    // Validar y formatear timestamp
    if (empty($timestamp)) {
        $timestamp = date('Y-m-d H:i:s');
    } else {
        // Convertir ISO 8601 a MySQL datetime
        $timestamp = date('Y-m-d H:i:s', strtotime($timestamp));
    }
    
    return [
        'rating' => $rating,
        'conversation_id' => $conversationId,
        'timestamp' => $timestamp
    ];
}

// ========================================
// EJECUCIÓN PRINCIPAL
// ========================================

try {
    // Conectar a la base de datos
    $pdo = getDBConnection();
    
    if (!$pdo) {
        sendResponse(false, 'Error de conexión a la base de datos');
    }
    
    // Inicializar base de datos (crear tabla si no existe)
    if (!initializeDatabase($pdo)) {
        sendResponse(false, 'Error al inicializar la base de datos');
    }
    
    // Manejar solicitud GET para estadísticas
    if ($_SERVER['REQUEST_METHOD'] === 'GET') {
        $stats = getRatingStats($pdo);
        sendResponse(true, 'Estadísticas obtenidas exitosamente', ['stats' => $stats]);
    }
    
    // Validar datos de entrada
    $input = validateInput();
    
    if (!$input) {
        sendResponse(false, 'Método de solicitud no permitido. Use POST.');
    }
    
    // Guardar calificación
    $success = saveRating(
        $pdo,
        $input['conversation_id'],
        $input['rating'],
        $input['timestamp']
    );
    
    if ($success) {
        // Obtener estadísticas actualizadas
        $stats = getRatingStats($pdo);
        
        sendResponse(
            true, 
            'Calificación guardada exitosamente',
            [
                'rating_saved' => $input['rating'],
                'conversation_id' => $input['conversation_id'],
                'stats' => $stats
            ]
        );
    } else {
        sendResponse(false, 'Error al guardar la calificación');
    }
    
} catch (Exception $e) {
    error_log("Error general: " . $e->getMessage());
    sendResponse(false, 'Error interno del servidor');
}

?>